package gov.va.med.mhv.usermgmt.service;

import gov.va.med.mhv.usermgmt.enumeration.OrganizationTypeEnumeration;
import gov.va.med.mhv.usermgmt.messages.EmployeeManagementMessages;
import gov.va.med.mhv.usermgmt.service.delegate.EmployeeRoleManagementServiceDelegate;
import gov.va.med.mhv.usermgmt.service.delegate.EntityMaintenanceServiceDelegate;
import gov.va.med.mhv.usermgmt.service.delegate.ServiceDelegateFactory;
import gov.va.med.mhv.usermgmt.transfer.Employee;
import gov.va.med.mhv.usermgmt.transfer.EmployeeOrganizationRole;
import gov.va.med.mhv.usermgmt.transfer.EmployeeOrganizationRoleHistory;
import gov.va.med.mhv.usermgmt.transfer.Organization;
import gov.va.med.mhv.usermgmt.transfer.OrganizationTypeRole;
import gov.va.med.mhv.usermgmt.transfer.Role;
import gov.va.med.mhv.usermgmt.transfer.TransferObjectFactory;
import junit.framework.TestCase;

import org.tigris.atlas.messages.Message;

public class TestEmployeeRoleManagement extends TestCase {
	private EntityMaintenanceServiceDelegate emDelegate = ServiceDelegateFactory.createEntityMaintenanceServiceDelegate();
	private EmployeeRoleManagementServiceDelegate sDelegate = ServiceDelegateFactory.createEmployeeRoleManagementServiceDelegate();
		
	private Employee employeeNationalAdmin;
	private Employee employeeBeingManaged;
	private Organization nationalOrg;
	private Organization visnOrg;
	private Role roleAdministratorRole;
	private Role himsOfficerRole;
	private EmployeeOrganizationRole empOrgRole;
	private OrganizationTypeRole nationalOrgAdminRole;
	private OrganizationTypeRole nationalOrgHimsOfficerRole;
	private OrganizationTypeRole visnOrgAdminRole;
	private OrganizationTypeRole visnOrgHimsOfficerRole;
	
	//service responses that are heavily reused
	private EmployeeServiceResponse empServiceResponse;
	private RoleServiceResponse roleServiceResponse;
	private OrganizationServiceResponse orgServiceResponse;
	private EmployeeOrganizationRoleServiceResponse empOrgRoleServiceResponse;
	private OrganizationTypeRoleServiceResponse orgTypeRoleServiceResponse;
	
	private Message m;
			
	/**
	 * due to large quantity of test data, i am combining multiple test cases into this 1 method
	 * this means that as tests execute, the originial state of the database may be altered
	 * care needs to be taken when adding new tests to ensure that they are placed in right order
	 */
	public void testEmployeeRoleManagementTests(){
		//prior to running tests, make sure get orgs employee can manage query works as expected
		assertTrue(sDelegate.getOrganizationsEmployeeCanManage(employeeNationalAdmin).getOrganizations().contains(nationalOrg));
		assertTrue(sDelegate.getOrganizationsEmployeeCanManage(employeeNationalAdmin).getOrganizations().contains(visnOrg));
		assertTrue(sDelegate.getOrganizationsEmployeeCanManage(employeeBeingManaged).getOrganizations().size() == 0);
		
		//prior to running tests, make sure get roles at org employee can manage query works as expected
		assertTrue(sDelegate.getRolesAtOrganizationEmployeeCanManage(employeeNationalAdmin,nationalOrg).getRoles().size() == 1);
		assertTrue(sDelegate.getRolesAtOrganizationEmployeeCanManage(employeeNationalAdmin,nationalOrg).getRoles().contains(himsOfficerRole));
		assertTrue(sDelegate.getRolesAtOrganizationEmployeeCanManage(employeeBeingManaged,nationalOrg).getRoles().size() == 0);
		
		//these tests should fail, therefore won't change state of database
		securityTestNonAdminAssignAdmin();
		securityTestAdminAssignNonAdminAtChildOrg();
		securityTestAdminAssignAdminAtSameOrg();	
		
		//these tests will result in new roles being added - run last
		assignRoleSuccessTests();			
	}
	/**
	 * This test involves a role administrator trying to assign another role administrator at their same organization
	 * This should fail due to security violation - so no database changes will be commited
	 */
	private void securityTestAdminAssignAdminAtSameOrg() {
		empOrgRoleServiceResponse = sDelegate.assignEmployeeOrganizationRole(
				employeeNationalAdmin,employeeBeingManaged.getUserName(),nationalOrg,roleAdministratorRole);
		assertTrue( empOrgRoleServiceResponse.getMessages().hasErrorMessages() );
		m = (Message) empOrgRoleServiceResponse.getMessages().getErrorMessages().iterator().next();
		assertEquals( EmployeeManagementMessages.MANAGEROLES_SECURITY_ERROR, m.getKey() );
	}	
	/**
	 * This test involves a NON role administrator trying to assign a role administrator
	 * This should fail due to security violation - so no database changes will be commited
	 */
	private void securityTestNonAdminAssignAdmin() {
		empOrgRoleServiceResponse = sDelegate.assignEmployeeOrganizationRole(
				employeeBeingManaged,employeeNationalAdmin.getUserName(),visnOrg,roleAdministratorRole);
		assertTrue( empOrgRoleServiceResponse.getMessages().hasErrorMessages() );
		m = (Message) empOrgRoleServiceResponse.getMessages().getErrorMessages().iterator().next();
		assertEquals( EmployeeManagementMessages.MANAGEROLES_SECURITY_ERROR, m.getKey() );
	}
	/**
	 * This test involves a role administrator trying to assign a NON role administrator at a child org
	 * This should fail due to security violation - so no database changes will be commited
	 */
	private void securityTestAdminAssignNonAdminAtChildOrg() {
		empOrgRoleServiceResponse = sDelegate.assignEmployeeOrganizationRole(
				employeeNationalAdmin,employeeBeingManaged.getUserName(),visnOrg,himsOfficerRole);
		assertTrue( empOrgRoleServiceResponse.getMessages().hasErrorMessages() );
		m = (Message) empOrgRoleServiceResponse.getMessages().getErrorMessages().iterator().next();
		assertEquals( EmployeeManagementMessages.MANAGEROLES_SECURITY_ERROR, m.getKey() );
	}
	/**
	 * This test involves numerous tests after a successfull role assignment where an admin assigns another admin at a child org
	 * This test will succeed, numerous database changes will be committed, don't run tests after this that can't handle these changes
	 */
	private void assignRoleSuccessTests() {
		//assign the employee we are managing as a visn role admin
		empOrgRoleServiceResponse = sDelegate.assignEmployeeOrganizationRole(
				employeeNationalAdmin,employeeBeingManaged.getUserName(),visnOrg,roleAdministratorRole);
		assertFalse( empOrgRoleServiceResponse.getMessages().hasErrorMessages() );
		assertTrue( empOrgRoleServiceResponse.getMessages().hasInformationalMessages() );	
		m = (Message) empOrgRoleServiceResponse.getMessages().getInformationalMessages().iterator().next();
		assertEquals( EmployeeManagementMessages.MANAGEROLES_ASSIGN_SUCCESS, m.getKey() );
		
		//test that the role history was created
		assertTrue(sDelegate.getRoleHistoriesForEmployee(employeeBeingManaged).getEmployeeOrganizationRoleHistorys().size() == 1);
		
		//ensure that this role shows up as one that our admin can now manage
		empOrgRole = empOrgRoleServiceResponse.getEmployeeOrganizationRole();
		assertTrue(sDelegate.getEmployeeOrganizationRolesEmployeeCanManage(employeeNationalAdmin).getEmployeeOrganizationRoles().size() == 1);
		assertTrue(sDelegate.getEmployeeOrganizationRolesEmployeeCanManage(employeeNationalAdmin).getEmployeeOrganizationRoles().contains(empOrgRole));
				
		//ensure that this role shows up as assigned to our employee we are managing	
		assertTrue(sDelegate.getEmployeeOrganizationRolesAssignedToEmployee(employeeBeingManaged).getEmployeeOrganizationRoles().size() == 1);
		assertTrue(sDelegate.getEmployeeOrganizationRolesAssignedToEmployee(employeeBeingManaged).getEmployeeOrganizationRoles().contains(empOrgRole));
		
		//ensure get organizations of type where employee has role service will work for this new assignment
		assertTrue(sDelegate.getOrganizationsOfTypeWhereEmployeeHasRole(employeeBeingManaged,visnOrg.getTypeOfOrganization(),roleAdministratorRole.getName()).getOrganizations().size() == 1);
		assertTrue(sDelegate.getOrganizationsOfTypeWhereEmployeeHasRole(employeeBeingManaged,visnOrg.getTypeOfOrganization(),roleAdministratorRole.getName()).getOrganizations().contains(visnOrg));
		
		//attempt to readd this employee to visn admin role - ensure we can't assign same role to same employee at same org 2 times
		empOrgRoleServiceResponse = sDelegate.assignEmployeeOrganizationRole(
				employeeNationalAdmin,employeeBeingManaged.getUserName(),visnOrg,roleAdministratorRole);
		assertTrue( empOrgRoleServiceResponse.getEmployeeOrganizationRole().getMessages().hasErrorMessages() );
		m = (Message) empOrgRoleServiceResponse.getEmployeeOrganizationRole().getMessages().getErrorMessages().iterator().next();
		assertEquals( EmployeeManagementMessages.MANAGEROLES_DUPLICATE_ASSIGNMENT, m.getKey() );
		
		//ensure we can't deactivate only user in this role	
		empOrgRole.setActive(false);
		empOrgRoleServiceResponse = sDelegate.updateEmployeeOrganizationRole(employeeNationalAdmin,empOrgRole);
		assertTrue( empOrgRoleServiceResponse.getEmployeeOrganizationRole().getMessages().hasErrorMessages() );
		m = (Message) empOrgRoleServiceResponse.getEmployeeOrganizationRole().getMessages().getErrorMessages().iterator().next();
		assertEquals( EmployeeManagementMessages.MANAGEROLES_CANT_REMOVE_LAST, m.getKey() );
		
		//now assign the employeeNationalAdmin as this same visn role admin
		empOrgRoleServiceResponse = sDelegate.assignEmployeeOrganizationRole(
				employeeNationalAdmin,employeeNationalAdmin.getUserName(),visnOrg,roleAdministratorRole);
		assertFalse( empOrgRoleServiceResponse.getMessages().hasErrorMessages() );
		assertTrue( empOrgRoleServiceResponse.getMessages().hasInformationalMessages() );	
		m = (Message) empOrgRoleServiceResponse.getMessages().getInformationalMessages().iterator().next();
		assertEquals( EmployeeManagementMessages.MANAGEROLES_ASSIGN_SUCCESS, m.getKey() );
		
		//now retry the deactivate, it should work this time cause there are 2 people in the role
		empOrgRoleServiceResponse = sDelegate.updateEmployeeOrganizationRole(employeeNationalAdmin,empOrgRole);
		assertFalse( empOrgRoleServiceResponse.getMessages().hasErrorMessages() );
	}		
	/**
	 * This utility method helps the tear down routine delete everything there is about an employee so we can remove then entirely
	 * @param employee
	 */
	private void deleteEmployee(Employee employee){				
		//search and destroy role histories AND roles for both employee, then whack the employee entity itself
		
		//deleate employee's role historys
		EmployeeOrganizationRoleHistory history;
		for(Object obj: sDelegate.getRoleHistoriesForEmployee(employee).getEmployeeOrganizationRoleHistorys()){
			history = (EmployeeOrganizationRoleHistory)obj;
			assertFalse(emDelegate.delete(history.getEmployeeOrganizationRoleHistoryPK()).getMessages().hasErrorMessages());	
		}
		
		//delete employee's roles
		EmployeeOrganizationRole role;
		for(Object obj: sDelegate.getEmployeeOrganizationRolesAssignedToEmployee(employee).getEmployeeOrganizationRoles()){
			role = (EmployeeOrganizationRole)obj;
			assertFalse(emDelegate.delete(role.getEmployeeOrganizationRolePK()).getMessages().hasErrorMessages());	
		}
		
		//now delete employee
		assertFalse(emDelegate.delete(employee.getEmployeePK()).getMessages().hasErrorMessages());
	}	
	/* (non-Javadoc)
	 * @see junit.framework.TestCase#setUp()
	 * test data we setup is
	 *	 2 employees, 1 seeded as national admin, and 1 being managed
	 *	 2 orgs, national and visn
	 *	 2 roles, the role admin, hims officer
	 *	 and lookup data so that service can see which roles are available at which org
	 */
	public void setUp() throws Exception {
		super.setUp();
			
		//save employeeDoingManagement
		employeeNationalAdmin = TransferObjectFactory.createEmployee();
		employeeNationalAdmin.setUserName("natadmin");
		employeeNationalAdmin.setFirstName("natadmin");
		employeeNationalAdmin.setLastName("natadmin");		
		empServiceResponse = emDelegate.save(employeeNationalAdmin);
		assertFalse(empServiceResponse.getMessages().hasErrorMessages());
		assertFalse(empServiceResponse.getEmployee().getMessages().hasErrorMessages());
		employeeNationalAdmin = empServiceResponse.getEmployee();	
		
		//save employeeBeingManaged
		employeeBeingManaged = TransferObjectFactory.createEmployee();
		employeeBeingManaged.setUserName("employee");
		employeeBeingManaged.setFirstName("employee");
		employeeBeingManaged.setLastName("employee");		
		empServiceResponse = emDelegate.save(employeeBeingManaged);
		assertFalse(empServiceResponse.getMessages().hasErrorMessages());
		assertFalse(empServiceResponse.getEmployee().getMessages().hasErrorMessages());
		employeeBeingManaged = empServiceResponse.getEmployee();
		
		//save the nationalOrg
		nationalOrg = TransferObjectFactory.createOrganization();
		nationalOrg.setName("National");
		nationalOrg.setTypeOfOrganization(OrganizationTypeEnumeration.getEnum(OrganizationTypeEnumeration.NATIONAL));
		orgServiceResponse = emDelegate.save(nationalOrg);
		assertFalse(orgServiceResponse.getMessages().hasErrorMessages());
		assertFalse(orgServiceResponse.getOrganization().getMessages().hasErrorMessages());
		nationalOrg = orgServiceResponse.getOrganization();
		
		//save the visnOrg
		visnOrg = TransferObjectFactory.createOrganization();
		visnOrg.setName("Visn 1");
		visnOrg.setTypeOfOrganization(OrganizationTypeEnumeration.getEnum(OrganizationTypeEnumeration.VISN));
		visnOrg.setParentOrganization(nationalOrg);
		orgServiceResponse = emDelegate.save(visnOrg);
		assertFalse(orgServiceResponse.getMessages().hasErrorMessages());
		assertFalse(orgServiceResponse.getOrganization().getMessages().hasErrorMessages());
		visnOrg = orgServiceResponse.getOrganization();
						
		//save roleAdministratorRole
		roleAdministratorRole = TransferObjectFactory.createRole();
		roleAdministratorRole.setName("Role Administrator");
		roleAdministratorRole.setSecurityAdmin(true);
		roleServiceResponse = emDelegate.save(roleAdministratorRole);
		assertFalse(roleServiceResponse.getMessages().hasErrorMessages());
		assertFalse(roleServiceResponse.getRole().getMessages().hasErrorMessages());
		roleAdministratorRole = roleServiceResponse.getRole();
		
		//save himsPrivacyOfficerRole
		himsOfficerRole = TransferObjectFactory.createRole();
		himsOfficerRole.setName("HIMS Privacy Officer");
		himsOfficerRole.setSecurityAdmin(false);
		roleServiceResponse = emDelegate.save(himsOfficerRole);
		assertFalse(roleServiceResponse.getMessages().hasErrorMessages());
		assertFalse(roleServiceResponse.getRole().getMessages().hasErrorMessages());
		himsOfficerRole = roleServiceResponse.getRole();
							
		//save empOrgRole (manually assign employeeDoingManagement as National Role Administrator)
		empOrgRole = TransferObjectFactory.createEmployeeOrganizationRole();
		empOrgRole.setEmployee(employeeNationalAdmin);
		empOrgRole.setOrganization(nationalOrg);
		empOrgRole.setRole(roleAdministratorRole);
		empOrgRole.setActive(true);
		empOrgRoleServiceResponse = emDelegate.save(empOrgRole);
		assertFalse(empOrgRoleServiceResponse.getMessages().hasErrorMessages());
		assertFalse(empOrgRoleServiceResponse.getEmployeeOrganizationRole().getMessages().hasErrorMessages());
		empOrgRole = empOrgRoleServiceResponse.getEmployeeOrganizationRole();	
		
		//setup org type role - national role admin
		nationalOrgAdminRole = TransferObjectFactory.createOrganizationTypeRole();
		nationalOrgAdminRole.setTypeOfOrganization(OrganizationTypeEnumeration.getEnum(OrganizationTypeEnumeration.NATIONAL));
		nationalOrgAdminRole.setRole(roleAdministratorRole);
		orgTypeRoleServiceResponse = emDelegate.save(nationalOrgAdminRole);
		assertFalse(orgTypeRoleServiceResponse.getMessages().hasErrorMessages());
		assertFalse(orgTypeRoleServiceResponse.getOrganizationTypeRole().getMessages().hasErrorMessages());
		nationalOrgAdminRole = orgTypeRoleServiceResponse.getOrganizationTypeRole();
		
		//setup org type role - national hims officer
		nationalOrgHimsOfficerRole = TransferObjectFactory.createOrganizationTypeRole();
		nationalOrgHimsOfficerRole.setTypeOfOrganization(OrganizationTypeEnumeration.getEnum(OrganizationTypeEnumeration.NATIONAL));
		nationalOrgHimsOfficerRole.setRole(himsOfficerRole);
		orgTypeRoleServiceResponse = emDelegate.save(nationalOrgHimsOfficerRole);
		assertFalse(orgTypeRoleServiceResponse.getMessages().hasErrorMessages());
		assertFalse(orgTypeRoleServiceResponse.getOrganizationTypeRole().getMessages().hasErrorMessages());
		nationalOrgHimsOfficerRole = orgTypeRoleServiceResponse.getOrganizationTypeRole();
		
		//setup org type role - visn role administrator
		visnOrgAdminRole = TransferObjectFactory.createOrganizationTypeRole();
		visnOrgAdminRole.setTypeOfOrganization(OrganizationTypeEnumeration.getEnum(OrganizationTypeEnumeration.VISN));
		visnOrgAdminRole.setRole(roleAdministratorRole);
		orgTypeRoleServiceResponse = emDelegate.save(visnOrgAdminRole);
		assertFalse(orgTypeRoleServiceResponse.getMessages().hasErrorMessages());
		assertFalse(orgTypeRoleServiceResponse.getOrganizationTypeRole().getMessages().hasErrorMessages());
		visnOrgAdminRole = orgTypeRoleServiceResponse.getOrganizationTypeRole();
		
		//setup org type role - visn hims officer
		visnOrgHimsOfficerRole = TransferObjectFactory.createOrganizationTypeRole();
		visnOrgHimsOfficerRole.setTypeOfOrganization(OrganizationTypeEnumeration.getEnum(OrganizationTypeEnumeration.VISN));
		visnOrgHimsOfficerRole.setRole(himsOfficerRole);
		orgTypeRoleServiceResponse = emDelegate.save(visnOrgHimsOfficerRole);
		assertFalse(orgTypeRoleServiceResponse.getMessages().hasErrorMessages());
		assertFalse(orgTypeRoleServiceResponse.getOrganizationTypeRole().getMessages().hasErrorMessages());
		visnOrgHimsOfficerRole = orgTypeRoleServiceResponse.getOrganizationTypeRole();						
	}	
	/* (non-Javadoc)
	 * @see junit.framework.TestCase#tearDown()
	 * delete all data we created during execution of these tests
	 */
	public void tearDown() throws Exception {
		deleteEmployee(employeeBeingManaged);
		deleteEmployee(employeeNationalAdmin);
		
		//delete orgs
		assertFalse(emDelegate.delete(visnOrg.getOrganizationPK()).getMessages().hasErrorMessages());	
		assertFalse(emDelegate.delete(nationalOrg.getOrganizationPK()).getMessages().hasErrorMessages());		
		
		//delete org type roles
		assertFalse(emDelegate.delete(nationalOrgAdminRole.getOrganizationTypeRolePK()).getMessages().hasErrorMessages());		
		assertFalse(emDelegate.delete(nationalOrgHimsOfficerRole.getOrganizationTypeRolePK()).getMessages().hasErrorMessages());		
		assertFalse(emDelegate.delete(visnOrgAdminRole.getOrganizationTypeRolePK()).getMessages().hasErrorMessages());		
		assertFalse(emDelegate.delete(visnOrgHimsOfficerRole.getOrganizationTypeRolePK()).getMessages().hasErrorMessages());		
						
		//delete roles
		assertFalse(emDelegate.delete(roleAdministratorRole.getRolePK()).getMessages().hasErrorMessages());	
		assertFalse(emDelegate.delete(himsOfficerRole.getRolePK()).getMessages().hasErrorMessages());	
	}
}